﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library.Enums;
    using Utilities;
    using Hims.Shared.UserModels.AssessmentType;
    using Hims.Shared.UserModels.Common;
    using Hims.Api.Models;

    /// <inheritdoc />
    [Authorize]
    [Route("api/assessment-type")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class AssessmentTypeController : BaseController
    {
        /// <summary>
        /// The assessment type services.
        /// </summary>
        private readonly IAssessmentTypeService assessmentTypeService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public AssessmentTypeController(IAssessmentTypeService assessmentTypeService, IAuditLogService auditLogServices)
        {
            this.assessmentTypeService = assessmentTypeService;
            this.auditLogServices = auditLogServices;
        }

        /// <summary>
        /// The fetch coupons.
        /// </summary>
        /// <param name="model">
        /// The coupon filter model.
        /// </param>
        /// <returns>
        /// The list of coupons.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<ViewModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] FilterModel model)
        {
            try
            {
                model = (FilterModel)EmptyFilter.Handler(model);
                var assessmentType = await this.assessmentTypeService.FetchAsync(model);
                return assessmentType == null ? this.ServerError() : this.Success(assessmentType);

            }
            catch (Exception ex)
            {

            }
            return default;
        }


        /// <summary>
        /// The add coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon added successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] InsertModel model, [FromHeader] LocationHeader header)
        {
            model = (InsertModel)EmptyFilter.Handler(model);
            var response = await this.assessmentTypeService.AddAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given Assessment Type has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.AssessmentType,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LocationId = Convert.ToInt32(header.LocationId),
                LogDescription = $" <b>{ model.CreatedByName}</b> has <b>added</b> Assessment Type <b>{model.Name}</b> on { DateTime.UtcNow.AddMinutes(330) }"
              
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("Assessment Type has been added successfully.");
        }

        /// <summary>
        /// The update coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// 
        /// 
        /// 
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon updated successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] UpdateModel model, [FromHeader] LocationHeader header)
        {
            model = (UpdateModel)EmptyFilter.Handler(model);
            var response = await this.assessmentTypeService.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given Assessment Type has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.AssessmentType,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LocationId = Convert.ToInt32(header.LocationId),
                LogDescription = $" <b>{ model.ModifiedByName }</b> has <b>updated</b> Assessment Type <b>{model.Name}</b> on { DateTime.UtcNow.AddMinutes(330) }"
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("Assessment Type has been updated successfully.");
        }

        /// <summary>
        /// The delete specialization.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <param name="header">
        /// The header.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Specialization deleted successfully.
        /// - 409 - Specialization can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] DeleteModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                model = (DeleteModel)EmptyFilter.Handler(model);
                var name = await this.assessmentTypeService.FindNameByAssessmentTypeId(model.AssessmentTypeId);
                var response = await this.assessmentTypeService.DeleteAsync(model);
                if (response == 0)
                {
                    return this.ServerError();
                }

                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.AssessmentType,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LocationId = Convert.ToInt32(header.LocationId),
                    LogDescription = $" <b>{ model.ModifiedByName }</b> has <b>deleted</b> Assessment Type <b>{name}</b> on { DateTime.UtcNow.AddMinutes(330) }"
                };
                await this.auditLogServices.LogAsync(auditLogModel);

                return this.Success("Assessment Type has been deleted successfully.");
            }
            catch (NpgsqlException exception)
            {
                if (exception.Message.Contains("violates foreign key constraint"))
                {
                    return this.Conflict("Assessment Type can't be deleted. Please contact Administrator.");
                }

                return this.ServerError();
            }
        }       
    }
}